<?php
// public/inc/auth.php
// Debe cargarse DESPUÉS de public/inc/bootstrap.php en cada página.
// Aun así, este helper pdo() garantiza que siempre obtendremos una conexión.

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

/** Devuelve una instancia válida de PDO siempre. */
if (!function_exists('pdo')) {
  function pdo(): PDO {
    // 1) Si bootstrap ya creó $pdo, úsalo
    if (!empty($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
      return $GLOBALS['pdo'];
    }
    // 2) Si existe la clase Database de bootstrap, creamos una nueva
    if (class_exists('Database')) {
      $db = new Database();
      $GLOBALS['pdo'] = $db->pdo;
      return $GLOBALS['pdo'];
    }
    // 3) Intento de incluir bootstrap como último recurso
    $bootstrap = __DIR__ . '/bootstrap.php';
    if (is_file($bootstrap)) {
      require_once $bootstrap;
      if (!empty($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
        return $GLOBALS['pdo'];
      }
    }
    throw new RuntimeException('No se pudo inicializar PDO (revisa bootstrap.php).');
  }
}

if (!function_exists('is_logged_in')) {
  function is_logged_in(): bool {
    return !empty($_SESSION['user_id']);
  }
}

if (!function_exists('current_user')) {
  function current_user(): ?array {
    if (!empty($_SESSION['user'])) return $_SESSION['user'];
    if (empty($_SESSION['user_id'])) return null;

    $st = pdo()->prepare("SELECT id, username, rol, activo FROM users WHERE id = ? LIMIT 1");
    $st->execute([$_SESSION['user_id']]);
    $u = $st->fetch(PDO::FETCH_ASSOC);
    if ($u) {
      $_SESSION['user'] = $u;
      return $u;
    }
    return null;
  }
}

if (!function_exists('require_login')) {
  function require_login(): void {
    if (!is_logged_in()) {
      header('Location: /farmacia/public/login.php');
      exit;
    }
  }
}

if (!function_exists('logout')) {
  function logout(): void {
    $_SESSION = [];
    if (ini_get('session.use_cookies')) {
      $p = session_get_cookie_params();
      setcookie(session_name(), '', time()-42000, $p['path'], $p['domain'], $p['secure'], $p['httponly']);
    }
    session_destroy();
  }
}

if (!function_exists('attempt_login')) {
  function attempt_login(string $username, string $password): bool {
    $sql = "
      SELECT id, username, rol, activo,
             COALESCE(password_hash, pass_hash) AS ph
      FROM users
      WHERE username = ?
      LIMIT 1
    ";
    $st = pdo()->prepare($sql);
    $st->execute([$username]);
    $u = $st->fetch(PDO::FETCH_ASSOC);

    if (!$u || empty($u['ph'])) return false;
    if (!password_verify($password, $u['ph'])) return false;
    if (isset($u['activo']) && (int)$u['activo'] === 0) return false;

    $_SESSION['user_id'] = (int)$u['id'];
    $_SESSION['user'] = [
      'id' => (int)$u['id'],
      'username' => $u['username'],
      'rol' => $u['rol'] ?? null,
      'activo' => $u['activo'] ?? 1,
    ];
    return true;
  }
}
