<?php
require __DIR__.'/../../inc/bootstrap.php';

function ymd($v) {
  $v = trim((string)$v);
  if ($v==='' ) return '';
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/',$v)) return $v;
  if (preg_match('/^(\d{2})-(\d{2})-(\d{4})$/',$v,$m)) return "{$m[3]}-{$m[2]}-{$m[1]}";
  return $v;
}

// Filtros
$f_desde = ymd($_GET['desde'] ?? '');
$f_hasta = ymd($_GET['hasta'] ?? '');

$where = []; $params = [];
if ($f_desde !== '') { $where[] = "p.fecha_elaboracion >= ?"; $params[] = $f_desde; }
if ($f_hasta !== '') { $where[] = "p.fecha_elaboracion <= ?"; $params[] = $f_hasta; }
$sqlWhere = $where ? "WHERE ".implode(" AND ", $where) : "";

// Traer preparados + doctor/paciente
$sql = "SELECT p.*, d.nombre AS doc, pa.nombre AS pac
        FROM preparados p
        JOIN doctores d   ON d.id=p.id_doctor
        JOIN pacientes pa ON pa.id=p.id_paciente
        $sqlWhere
        ORDER BY p.fecha_elaboracion DESC, p.id DESC
        LIMIT 1000";
$prep = $pdo->prepare($sql); $prep->execute($params); $rows = $prep->fetchAll();

// Helpers conversión a gramos (o g aproximados cuando aplica)
function gramos_por_componente($forma, $um, $cantidad, $total) {
  $forma = (string)$forma; $um = (string)$um;
  $cantidad = (int)$cantidad; $total = (int)$total;
  if ($cantidad <= 0 || $total <= 0) return 0.0;

  // Cápsulas: cantidad es por cápsula (mg/ug/UI) => convertir a g y multiplicar por n° cápsulas
  if ($forma === 'cápsulas') {
    if ($um === 'mg') return ($cantidad / 1000.0) * $total;
    if ($um === 'ug') return ($cantidad / 1000000.0) * $total;
    if ($um === 'UI') return 0.0; // sin conversión definida
    // fallback si ya viene en g
    return (float)$cantidad;
  }

  // Cremas / Gel / Ungüento: % del total (g)
  if (in_array($forma, ['cremas','Gel','Ungüento'])) {
    if ($um === '%') return ($cantidad * $total) / 100.0;
    return (float)$cantidad; // fallback si ingresan g
  }

  // Locion / laca / shampoo / spray: % del total (ml) ~ g aprox densidad 1
  if (in_array($forma, ['Locion','laca','shampoo','spray'])) {
    if ($um === '%') return ($cantidad * $total) / 100.0;
    return (float)$cantidad;
  }

  // Jarabe / gotas: mg/ml -> g totales = (mg/ml * ml_total)/1000 ; % -> (ml_total * % / 100) ~ g
  if (in_array($forma, ['Jarabe','gotas'])) {
    if ($um === 'mg/ml') return ($cantidad * $total) / 1000.0;
    if ($um === '%')     return ($cantidad * $total) / 100.0;
    return (float)$cantidad;
  }

  // Supositorios: mg por supositorio -> g totales
  if ($forma === 'supositorios') {
    if ($um === 'mg') return ($cantidad / 1000.0) * $total;
    return (float)$cantidad;
  }

  // Polvo: puede venir en mg (tratar como g totales ya provistos) o %
  if ($forma === 'polvo') {
    if ($um === 'mg') return ($cantidad / 1000.0);
    if ($um === '%')  return ($cantidad * $total) / 100.0;
    return (float)$cantidad;
  }

  // Fallback
  return (float)$cantidad;
}

function ingreso_preparado($row) {
  // Prioriza valor_real si existe; si no, usa valor_calculado
  $vr = (int)($row['valor_real'] ?? 0);
  if ($vr > 0) return $vr;
  return (int)($row['valor_calculado'] ?? 0);
}

// Pre-calcular costos por preparado
$datos = [];
if ($rows) {
  // Traemos todos los componentes de estos preparados de una vez
  $ids = array_map(fn($r)=> (int)$r['id'], $rows);
  $in  = implode(',', array_fill(0, count($ids), '?'));
  $stmt = $pdo->prepare("SELECT id_preparado, nombre_componente, unidad_medida, cantidad, compra_g, venta_g FROM componentes_preparado WHERE id_preparado IN ($in) ORDER BY id_preparado ASC, id ASC");
  $stmt->execute($ids);
  $byPrep = [];
  foreach ($stmt->fetchAll() as $c) {
    $byPrep[(int)$c['id_preparado']][] = $c;
  }

  foreach ($rows as $r) {
    $pid   = (int)$r['id'];
    $forma = (string)$r['forma_farmaceutica'];
    $total = (int)$r['total'];
    $comps = $byPrep[$pid] ?? [];

    $costo = 0.0;
    foreach ($comps as $c) {
      $g = gramos_por_componente($forma, $c['unidad_medida'], (int)$c['cantidad'], $total);
      $costo += ((float)$c['compra_g']) * $g;
    }

    $ingreso = ingreso_preparado($r);
    $margen  = $ingreso - (int)round($costo);
    $pct     = $ingreso > 0 ? round(($margen / $ingreso) * 100) : 0;

    $datos[] = [
      'id'      => $pid,
      'fecha'   => $r['fecha_elaboracion'],
      'num'     => $r['numero_externo'],
      'pac'     => $r['pac'],
      'doc'     => $r['doc'],
      'forma'   => $forma,
      'ingreso' => $ingreso,
      'costo'   => (int)round($costo),
      'margen'  => $margen,
      'pct'     => $pct,
    ];
  }
}
?>
<!doctype html><html lang="es"><head>
<meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<link rel="stylesheet" href="../../assets/styles.css">
<link rel="stylesheet" href="../../assets/theme.css">
<title>Reporte de margen</title>
<style>
  .totales { display:flex; gap:16px; flex-wrap:wrap; margin:12px 0; }
  .pill { padding:8px 12px; background:#eef6ff; border:1px solid #cfe2ff; border-radius:8px; }
  .pct-pos { color:#138a07; font-weight:600; }
  .pct-neg { color:#b91c1c; font-weight:600; }
</style>
</head><body>
<div class="container mini">
  <div class="card theme-prep">
    <div style="display:flex;gap:8px;align-items:center;justify-content:space-between">
      <a class="btn" href="list.php">← Volver a preparados</a>
      <div>
        <a class="btn" href="../../index.php">🏠 Inicio</a>
      </div>
    </div>
    <h2>Reporte de margen</h2>

    <!-- Filtros -->
    <form class="grid" style="grid-template-columns: repeat(5,1fr); align-items:end; gap:12px">
      <div>
        <div class="label">Desde</div>
        <input class="input" type="date" name="desde" value="<?=h($f_desde)?>">
      </div>
      <div>
        <div class="label">Hasta</div>
        <input class="input" type="date" name="hasta" value="<?=h($f_hasta)?>">
      </div>
      <div>
        <button class="btn" type="submit">🔎 Filtrar</button>
      </div>
      <div>
        <a class="btn" href="reporte_margen.php">❌ Limpiar</a>
      </div>
      <div>
        <button class="btn" type="button" onclick="exportCSV()">⬇️ Exportar CSV</button>
      </div>
    </form>

    <?php
      $sumIng = array_sum(array_column($datos,'ingreso'));
      $sumCost= array_sum(array_column($datos,'costo'));
      $sumMar = array_sum(array_column($datos,'margen'));
      $sumPct = $sumIng > 0 ? round(($sumMar / $sumIng) * 100) : 0;
    ?>
    <div class="totales">
      <div class="pill"><b>Ingresos:</b> $<?= number_format($sumIng,0,',','.') ?></div>
      <div class="pill"><b>Costos:</b> $<?= number_format($sumCost,0,',','.') ?></div>
      <div class="pill"><b>Margen:</b> $<?= number_format($sumMar,0,',','.') ?></div>
      <div class="pill"><b>Margen %:</b> <span class="<?= $sumPct>=0?'pct-pos':'pct-neg' ?>"><?= $sumPct ?>%</span></div>
    </div>

    <table class="table">
      <thead>
        <tr>
          <th>Fecha</th><th>N° externo</th><th>Forma</th><th>Paciente</th><th>Doc</th>
          <th style="text-align:right">Ingreso</th>
          <th style="text-align:right">Costo</th>
          <th style="text-align:right">Margen</th>
          <th style="text-align:right">%</th>
          <th>Acciones</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($datos as $d): ?>
          <tr>
            <td><?=h(fmt_date($d['fecha']))?></td>
            <td><?=h($d['num'])?></td>
            <td><?=h($d['forma'])?></td>
            <td><?=h($d['pac'])?></td>
            <td><?=h($d['doc'])?></td>
            <td style="text-align:right">$<?=number_format($d['ingreso'],0,',','.')?></td>
            <td style="text-align:right">$<?=number_format($d['costo'],0,',','.')?></td>
            <td style="text-align:right">$<?=number_format($d['margen'],0,',','.')?></td>
            <td style="text-align:right"><?= (int)$d['pct'] ?>%</td>
            <td>
              <a class="btn" href="label_preview.php?id=<?=$d['id']?>">🏷️ Etiqueta</a>
              <a class="btn" href="edit.php?id=<?=$d['id']?>">✏️ Editar</a>
            </td>
          </tr>
        <?php endforeach; ?>
        <?php if (!$datos): ?>
          <tr><td colspan="10" style="text-align:center;color:#666">Sin resultados</td></tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>
</div>

<script>
function exportCSV(){
  const rows = [...document.querySelectorAll('table.table tbody tr')];
  if (!rows.length) return alert('No hay datos para exportar.');
  let csv = 'Fecha,N externo,Forma,Paciente,Doc,Ingreso,Costo,Margen,Porcentaje\\n';
  rows.forEach(tr => {
    const tds = tr.querySelectorAll('td');
    if (tds.length < 9) return;
    const vals = [
      tds[0].innerText.trim(),
      tds[1].innerText.trim(),
      tds[2].innerText.trim(),
      tds[3].innerText.trim(),
      tds[4].innerText.trim(),
      tds[5].innerText.replace(/[^0-9-]/g,''),
      tds[6].innerText.replace(/[^0-9-]/g,''),
      tds[7].innerText.replace(/[^0-9-]/g,''),
      tds[8].innerText.replace(/[^0-9-]/g,''),
    ];
    csv += vals.map(v => `"${v}"`).join(',') + '\\n';
  });
  const blob = new Blob([csv], {type:'text/csv;charset=utf-8;'});
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url; a.download = 'reporte_margen.csv'; a.click();
  URL.revokeObjectURL(url);
}
</script>
</body></html>
